/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
describeWithFlags('basicLSTMCell', ALL_ENVS, () => {
    it('basicLSTMCell with batch=2', async () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = tf.randomNormal([4]);
        const forgetBias = tf.scalar(1.0);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        const [newC, newH] = tf.basicLSTMCell(forgetBias, lstmKernel, lstmBias, batchedData, batchedC, batchedH);
        const newCVals = await newC.array();
        const newHVals = await newH.array();
        expect(newCVals[0][0]).toEqual(newCVals[1][0]);
        expect(newHVals[0][0]).toEqual(newHVals[1][0]);
    });
    it('basicLSTMCell accepts a tensor-like object', async () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = [0, 0, 0, 0];
        const forgetBias = 1;
        const data = [[0, 0]]; // 1x2
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = [[0]]; // 1x1
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = [[0]]; // 1x1
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        const [newC, newH] = tf.basicLSTMCell(forgetBias, lstmKernel, lstmBias, batchedData, batchedC, batchedH);
        const newCVals = await newC.array();
        const newHVals = await newH.array();
        expect(newCVals[0][0]).toEqual(newCVals[1][0]);
        expect(newHVals[0][0]).toEqual(newHVals[1][0]);
    });
});
describeWithFlags('basicLSTMCell throws with non-tensor', ALL_ENVS, () => {
    it('input: forgetBias', () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = tf.randomNormal([4]);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        expect(() => tf.basicLSTMCell({}, lstmKernel, lstmBias, batchedData, batchedC, batchedH))
            .toThrowError(/Argument 'forgetBias' passed to 'basicLSTMCell' must be a Tensor/);
    });
    it('input: lstmKernel', () => {
        const lstmBias = tf.randomNormal([4]);
        const forgetBias = tf.scalar(1.0);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        expect(() => tf.basicLSTMCell(forgetBias, {}, lstmBias, batchedData, batchedC, batchedH))
            .toThrowError(/Argument 'lstmKernel' passed to 'basicLSTMCell' must be a Tensor/);
    });
    it('input: lstmBias', () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const forgetBias = tf.scalar(1.0);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        expect(() => tf.basicLSTMCell(forgetBias, lstmKernel, {}, batchedData, batchedC, batchedH))
            .toThrowError(/Argument 'lstmBias' passed to 'basicLSTMCell' must be a Tensor/);
    });
    it('input: data', () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = tf.randomNormal([4]);
        const forgetBias = tf.scalar(1.0);
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        expect(() => tf.basicLSTMCell(forgetBias, lstmKernel, lstmBias, {}, batchedC, batchedH))
            .toThrowError(/Argument 'data' passed to 'basicLSTMCell' must be a Tensor/);
    });
    it('input: c', () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = tf.randomNormal([4]);
        const forgetBias = tf.scalar(1.0);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const h = tf.randomNormal([1, 1]);
        const batchedH = tf.concat2d([h, h], 0); // 2x1
        expect(() => tf.basicLSTMCell(forgetBias, lstmKernel, lstmBias, batchedData, {}, batchedH))
            .toThrowError(/Argument 'c' passed to 'basicLSTMCell' must be a Tensor/);
    });
    it('input: h', () => {
        const lstmKernel = tf.randomNormal([3, 4]);
        const lstmBias = tf.randomNormal([4]);
        const forgetBias = tf.scalar(1.0);
        const data = tf.randomNormal([1, 2]);
        const batchedData = tf.concat2d([data, data], 0); // 2x2
        const c = tf.randomNormal([1, 1]);
        const batchedC = tf.concat2d([c, c], 0); // 2x1
        expect(() => tf.basicLSTMCell(forgetBias, lstmKernel, lstmBias, batchedData, batchedC, {}))
            .toThrowError(/Argument 'h' passed to 'basicLSTMCell' must be a Tensor/);
    });
});
//# sourceMappingURL=data:application/json;base64,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